<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Package;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PackageController extends Controller
{
    public function index(Request $request)
    {
        $query = Package::query();

        if ($request->has('category') && $request->category) {
            $query->where('category', $request->category);
        }

        $packages = $query->orderBy('sort_order')->paginate(10);
        return view('admin.packages.index', compact('packages'));
    }

    public function create()
    {
        return view('admin.packages.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'category' => 'required|in:best_kashmir,honeymoon,adventure,cultural',
            'duration_days' => 'required|integer|min:1',
            'duration_nights' => 'required|integer|min:0',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        $data = $request->except('image', 'location_tags', 'itinerary');
        $data['slug'] = Str::slug($request->title);

        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('images/packages'), $imageName);
            $data['image'] = 'images/packages/' . $imageName;
        }

        // Handle JSON fields
        if ($request->location_tags) {
            $data['location_tags'] = array_filter(explode(',', $request->location_tags));
        }
        if ($request->itinerary) {
            $data['itinerary'] = array_filter(explode("\n", $request->itinerary));
        }

        Package::create($data);

        return redirect()->route('admin.packages.index')->with('success', 'Package created successfully');
    }

    public function edit(Package $package)
    {
        return view('admin.packages.edit', compact('package'));
    }

    public function update(Request $request, Package $package)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'category' => 'required|in:best_kashmir,honeymoon,adventure,cultural',
            'duration_days' => 'required|integer|min:1',
            'duration_nights' => 'required|integer|min:0',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        $data = $request->except('image', 'location_tags', 'itinerary');

        // Only regenerate slug if title has changed
        if ($package->title !== $request->title) {
            $data['slug'] = Str::slug($request->title);
        } else {
            unset($data['slug']); // Don't update slug if title hasn't changed
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('images/packages'), $imageName);
            $data['image'] = 'images/packages/' . $imageName;
        }

        // Handle JSON fields
        if ($request->location_tags) {
            $data['location_tags'] = array_filter(explode(',', $request->location_tags));
        } else {
            $data['location_tags'] = [];
        }
        if ($request->itinerary) {
            $data['itinerary'] = array_filter(explode("\n", $request->itinerary));
        } else {
            $data['itinerary'] = [];
        }

        $package->update($data);

        return redirect()->route('admin.packages.index')->with('success', 'Package updated successfully');
    }

    public function destroy(Package $package)
    {
        $package->delete();
        return redirect()->route('admin.packages.index')->with('success', 'Package deleted successfully');
    }
}
