<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Enquiry;
use App\Models\FooterSetting;
use App\Models\SeoSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class EnquiryController extends Controller
{
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'mobile' => 'required|string|max:20',
            'package_name' => 'nullable|string|max:255',
            'persons' => 'nullable|integer|min:1',
            'booking_date' => 'nullable|date',
            'message' => 'nullable|string|max:2000',
        ]);

        // Create enquiry record
        $enquiry = Enquiry::create($validated);

        // Send email notification (if SMTP is configured)
        try {
            $footer = FooterSetting::getSettings();
            $adminEmail = $footer->email ?? 'info@flydivinetreasurekash.com';

            Mail::raw(
                "New enquiry received!\n\n" .
                "Name: {$enquiry->name}\n" .
                "Email: {$enquiry->email}\n" .
                "Mobile: {$enquiry->mobile}\n" .
                "Package: {$enquiry->package_name}\n" .
                "Persons: {$enquiry->persons}\n" .
                "Booking Date: {$enquiry->booking_date}\n" .
                "Message: {$enquiry->message}",
                function ($message) use ($enquiry, $adminEmail) {
                    $message->to($adminEmail)
                        ->subject('New Tour Enquiry from ' . $enquiry->name);
                }
            );
        } catch (\Exception $e) {
            // Email sending failed, but enquiry is saved
            // Log the error for debugging
            \Log::error('Email sending failed: ' . $e->getMessage());
        }

        return redirect()->route('thanks');
    }

    public function thanks()
    {
        $footer = FooterSetting::getSettings();
        $seoSettings = SeoSetting::getSettings();
        return view('frontend.pages.thanks', compact('footer', 'seoSettings'));
    }
}
